"use client"
import { useState, useEffect } from 'react';
import axios from 'axios';

export default function WatchThis() {
  const [searchQuery, setSearchQuery] = useState('');
  const [searchResults, setSearchResults] = useState([]);
  const [relatedMovies, setRelatedMovies] = useState([]);
  const [activeTab, setActiveTab] = useState('movies');
  const [relatedMoviesLoading, setRelatedMoviesLoading] = useState(false);

  const apiKey = process.env.NEXT_PUBLIC_WATCHTHIS_APIKEY;
  const tmdbApiKey = process.env.NEXT_PUBLIC_TMDB_APIKEY;

  const search = async () => {
    try {
      const type = activeTab === 'movies' ? 'movie' : 'tv';
      const response = await axios.get(`https://api.themoviedb.org/3/search/${type}?sort_by=popularity.desc`, {
        params: {
          api_key: tmdbApiKey,
          query: searchQuery,
        },
      });

      setSearchResults(response.data.results.slice(0, 4));
    } catch (error) {
      console.error('Error fetching search results:', error);
    }
  };

  useEffect(() => {
    search();
  }, [searchQuery, activeTab, tmdbApiKey]);

  const selectResult = async (result) => {
    try {
      const type = activeTab === 'movies' ? 'movie' : 'tv';

      setSearchQuery(result.name || result.title);
      setRelatedMoviesLoading(true);

      setSearchResults([])

      const response = await axios.get(`https://watchthis.p.rapidapi.com/api/v1/${type}`, {
        params: {
          ids: result.id,
        },
        headers: {
          'X-RapidAPI-Host': 'watchthis.p.rapidapi.com',
          'X-RapidAPI-Key': apiKey,
        },
      });

      setRelatedMovies(response.data.related.slice(0, 8));
      setSearchResults([])
    } catch (error) {
      console.error('Error fetching related movies:', error);
    } finally {
      setSearchResults([])
      setRelatedMoviesLoading(false);
    }
  };

  const filteredResults = searchResults.filter(result => result.vote_average >= 6);

  const switchTab = (tab) => {
    setActiveTab(tab);
    search();
  };

  return (
    <div>
      {/* Header */}
      <div className="flex flex-col items-center min-h-[90vh]">
        <div className="flex mt-8 bg-gray-100 rounded-full mb-[20vh]">
          <button
            onClick={() => switchTab('movies')}
            className={`p-2 px-4 rounded-full ease-in ease-out duration-300 ${activeTab === 'movies' ? 'bg-blue-500 text-white' : 'bg-transparent'}`}
          >
            Movies
          </button>
          <button
            onClick={() => switchTab('shows')}
            className={`p-2 px-4 rounded-full ease-in ease-out duration-300 ${activeTab === 'shows' ? 'bg-blue-500 text-white' : 'bg-transparent'}`}
          >
            Shows
          </button>
        </div>
        <h1 className="text-4xl md:text-5xl lg:text-6xl font-bold">
          Watch<span className="text-blue-500">This</span>
        </h1>
        <div className="mt-4 relative w-2/3 md:w-1/2">
          <div className="flex bg-slate-100 items-center rounded-lg border border-slate-400 md:text-xl">
            <i className="bx bx-search-alt md:text-slate-600 p-2"></i>
            <input
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              type="text"
              className="bg-transparent p-2 grow rounded-lg outline-none md:text-lg"
            />
          </div>
          {/* Autocomplete results */}
          {searchResults.length > 0 && (
            <div className="absolute top-full left-0 bg-white border border-slate-200 rounded-lg mt-2 w-full">
              <ul>
                {filteredResults.map((result) => (
                  <li
                    key={result.id}
                    className="p-1 px-4 pl-2 bg-slate-100 rounded-lg m-2"
                    onClick={() => selectResult(result)}
                  >
                    <div className="flex items-center">
                      <img
                        src={result.poster_path ? `https://image.tmdb.org/t/p/w92${result.poster_path}` : ''}
                        alt="Poster"
                        className="mr-2 w-8 rounded-md"
                      />
                      <div>
                        <p>{result.title || result.name} ({result.release_date ? result.release_date.substring(0, 4) : result.first_air_date.substring(0, 4)})</p>
                        <p>Rating: {result.vote_average}/10</p>
                      </div>
                    </div>
                  </li>
                ))}
              </ul>
            </div>
          )}
        </div>
        {/* Loading indicator for related movies */}
        {relatedMoviesLoading && (
          <div className="m-4 fixed top-0 left-0 text-xl flex justify-center items-center">
            <div role="status">
              <svg
                aria-hidden="true"
                className="w-6 h-6 text-gray-200 animate-spin fill-blue-600"
                viewBox="0 0 100 101"
                fill="none"
                xmlns="http://www.w3.org/2000/svg"
              >
                <path d="M100 50.5908C100 78.2051 77.6142 100.591 50 100.591C22.3858 100.591 0 78.2051 0 50.5908C0 22.9766 22.3858 0.59082 50 0.59082C77.6142 0.59082 100 22.9766 100 50.5908ZM9.08144 50.5908C9.08144 73.1895 27.4013 91.5094 50 91.5094C72.5987 91.5094 90.9186 73.1895 90.9186 50.5908C90.9186 27.9921 72.5987 9.67226 50 9.67226C27.4013 9.67226 9.08144 27.9921 9.08144 50.5908Z" fill="currentColor"/>
                <path d="M93.9676 39.0409C96.393 38.4038 97.8624 35.9116 97.0079 33.5539C95.2932 28.8227 92.871 24.3692 89.8167 20.348C85.8452 15.1192 80.8826 10.7238 75.2124 7.41289C69.5422 4.10194 63.2754 1.94025 56.7698 1.05124C51.7666 0.367541 46.6976 0.446843 41.7345 1.27873C39.2613 1.69328 37.813 4.19778 38.4501 6.62326C39.0873 9.04874 41.5694 10.4717 44.0505 10.1071C47.8511 9.54855 51.7191 9.52689 55.5402 10.0491C60.8642 10.7766 65.9928 12.5457 70.6331 15.2552C75.2735 17.9648 79.3347 21.5619 82.5849 25.841C84.9175 28.9121 86.7997 32.2913 88.1811 35.8758C89.083 38.2158 91.5421 39.6781 93.9676 39.0409Z" fill="currentFill"/>
              </svg>
              <span className="sr-only">Loading...</span>
            </div>
          </div>
        )}
        {/* Display related movies */}
        {relatedMovies.length > 0 && (
          <div className="mt-4 mx-auto w-2/3 md:w-1/2">
            <h2 className="text-xl font-semibold mb-2 capitalize">Related {activeTab}</h2>
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4">
              {relatedMovies.map((movie) => (
                <a key={movie.tmdb_id} href={`https://www.basicai.online/${activeTab}/${movie.tmdb_id}`} target="_blank">
                  <div className="flex flex-col items-center p-2 cursor-pointer">
                    <img
                      src={movie.tmdb_poster_path ? `https://image.tmdb.org/t/p/w200${movie.tmdb_poster_path}` : ''}
                      alt="Poster"
                      className="mr-2 w-full grow-1 h-56 object-cover rounded-md"
                    />
                    <div className="w-full max-w-full">
                      <p className="truncate">{movie.name || movie.title}</p>
                      <p
                        className={`font-semibold ${
                          movie.match_score <= 60 ? 'text-red-500' : movie.match_score <= 89 ? 'text-yellow-600' : 'text-green-500'
                        }`}
                      >
                        {movie.match_score}%
                      </p>
                    </div>
                  </div>
                </a>
              ))}
            </div>
          </div>
        )}
      </div>
      {/* Footer */}
      <div className="bg-slate-100 p-4 text-black text-center min-h-[10vh]">
        <p>
          Created with{' '}
          <a className="text-blue-500 font-semibold" href="https://rapidapi.com/vitalsx-apis-vitalsx-apis-default/api/watchthis">WatchThis API</a>
        </p>
      </div>
      </div>
    );
 }